<?php
/*
 * Plugin Name: MyInvoice2U e-Invoicing for WooCommerce
 * Plugin URI: https://www.myinvoice2u.com
 * Description: MyInvoice2U is a comprehensive e-invoicing service provider offering solutions tailored to businesses of all sizes in Malaysia, ensuring compliance with LHDN requirements. Our e-invoicing plugin for WooCommerce delivers seamless integration, ensuring that businesses effortlessly meet government e-invoicing mandates.
 * Version: 1.0.0
 * Author: MyInvoice2U
 * Author URI: https://www.myinvoice2u.com
 * Text Domain: mi2u-eInvoice
 * Domain Path: 
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

namespace MI2UINV;

use WC_Order_Query;
use WC_Payment_Gateway;
use Carbon;
use WC_Settings_Tax;
use WC_Tax;
use Date;
use GPBMetadata\Google\Type\Datetime;

if (! defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

add_filter('mi2u_einvoice', 'mi2u_add_autosumbit_class');
function mi2u_add_autosumbit_class($gateways)
{
	$gateways[] = 'MI2U_AUTOSUBMIT'; // your class name is here
	return $gateways;
}

//add_action('plugins_loaded', 'mi2u_init_submit_class');
add_action(
	'plugins_loaded',
	function () {
		MI2U_AUTOSUBMIT::get_instance();
	}
);

class mi2u_invoice
{
	protected $fillable = [
		'localDateTime',
		'storeCode',
		'invoiceCodeNumber',
		'issueMode',
		'originalInvoiceCodeNumber',
		'originalInvoiceUuid',
		'invoiceDate',
		'invoiceTime',
		'invoiceTypeCode',
		'invoiceCurrencyCode',
		'exchangeRate',
		'k1',
		'incoterms',
		'ftaInformation',
		'k2',
		'totalExcludingTax',
		'totalIncludingTax',
		'totalPayableAmount',
		'supplier',
		'buyer',
		'invoiceLineItemList',
		'taxTotal',
	];

	protected $attributes = [
		'localDateTime' => '',
		'storeCode' => '',
		'invoiceCodeNumber' => '',
		'issueMode' => '',
		'originalInvoiceCodeNumber' => '',
		'originalInvoiceUuid' => '',
		'invoiceDate' => '',
		'invoiceTime' => '',
		'invoiceTypeCode' => '',
		'invoiceCurrencyCode' => '',
		'exchangeRate' => '',
		'k1' => '',
		'incoterms' => '',
		'ftaInformation' => '',
		'k2' => '',
		'totalExcludingTax' => '0',
		'totalIncludingTax' => '0',
		'totalPayableAmount' => '0',
		'supplier' => null,
		'buyer' => null,
		'invoiceLineItemList' => [],
		'taxTotal' => null,
	];
}

// function mi2u_init_submit_class()
// {
class MI2U_AUTOSUBMIT
{
	private static $instances = array();

	/**
	 * Class constructor, more about it in Step 3
	 */
	public function __construct()
	{

		if (!function_exists('write_log')) {
			function write_log($log)
			{
				// if (true === WP_DEBUG) {
				// 	if (is_array($log) || is_object($log)) {
				// 		error_log(print_r($log, true));
				// 	} else {
				// 		error_log($log);
				// 	}
				// }
			}
		}

		add_action('init', array(&$this, 'mi2uinv_custom_action_function'));
		add_action('mi2u_check_order', array($this, 'mi2uinv_check_order_do_action'));

		$taxClass = WC_Tax::create_tax_class("Sales Tax: 01");
		$taxClass = WC_Tax::create_tax_class("Service Tax: 02");
		$taxClass = WC_Tax::create_tax_class("Tourism Tax: 03");
		$taxClass = WC_Tax::create_tax_class("High-Value Goods Tax: 04");
		$taxClass = WC_Tax::create_tax_class("Sales Tax on Low Value Goods: 05");

		add_action('admin_menu', array(&$this, 'mi2u_menu'));

		add_action('woocommerce_after_checkout_billing_form', array($this,  'mi2uinv_custom_checkout_field'));
		add_action(
			'woocommerce_checkout_update_order_meta',
			array($this, 'mi2uinv_save_custom_checkout_field_value_order_meta')
		);

		add_action('woocommerce_order_details_after_customer_details', array($this,  'mi2uinv_order_details'));
	}

	function mi2uinv_order_details($order)
	{
		$IDType = 'BRN';
		$IDValue = "";
		$TIN = "";

		$token = get_option('mi2uinv_login_email');
		$resp = $this->GetTransaction($token, $order->id);
		if ($resp['status'] == 'success') {
			$data = json_decode($resp['content']);
			if (count($data->records)) {
				$record = $data->records[0];
				if (isset($record->invoiceStatus) && $record->issuedFlag == "101" && $record->invoiceStatus == "101") {
					// Already issued E-Invoice
					$order->update_meta_data('up_ics', 'i');
					$order->save_meta_data();
				} else {
				}
			} else {
			}
		} else {
		}
		$up_ics = $order->get_meta('up_ics');
		if ($up_ics == 'i') {
?>
			<h3>E-Invoice</h3>
			<h2>Submitted</h2>
		<?php
			return;
		}
		$comp = $order->get_meta('mi2uinv_comp');
		if ($comp == '') {
			$comp = $order->get_billing_company();
		}
		?>

		<form method="post">
			<?php wp_nonce_field('mi2uinv_resubmit_action', 'mi2uinv_resubmit_nonce'); ?>
			<h3>E-Invoice</h3>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_comp', array(
					'type'        => 'text',
					'class'       => array('form-row-wide'),
					'label'       => __('Company name', 'mi2u-eInvoice'),
					'placeholder' => __('Company name', 'mi2u-eInvoice'),
					'required'    => false,
					'default' => $comp,
				)); ?>
			</div>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_tin', array(
					'type'        => 'text',
					'class'       => array('form-row-wide'),
					'label'       => __('TIN', 'mi2u-eInvoice'),
					'placeholder' => __('TIN', 'mi2u-eInvoice'),
					'required'    => false,
					'default' => 'EI00000000010'
				)); ?>
			</div>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_idtype', array(
					'type'        => 'radio',
					'class'       => array('form-row-wide'),
					'label'       => __('IDType', 'mi2u-eInvoice'),
					'placeholder' => __('IDType', 'mi2u-eInvoice'),
					'required'    => false,
					'options'     => array(
						'NRIC' => __('NRIC', 'mi2u-eInvoice'),
						'BRN' => __('BRN', 'mi2u-eInvoice'),
						'PASSPORT' => __('PASSPORT', 'mi2u-eInvoice'),
						'ARMY' => __('ARMY', 'mi2u-eInvoice'),
						'MyPR' => __('MyPR', 'mi2u-eInvoice'),
						'MyKAS' => __('MyKAS', 'mi2u-eInvoice'),
					),
					'default' => 'NRIC'
				)); ?>
			</div>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_idvalue', array(
					'type'        => 'text',
					'class'       => array('form-row-wide'),
					'label'       => __('IDValue', 'mi2u-eInvoice'),
					'placeholder' => __('IDValue', 'mi2u-eInvoice'),
					'required'    => false,
				)); ?>
			</div>
			<br>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_sst', array(
					'type'        => 'text',
					'class'       => array('form-row-wide'),
					'label'       => __('SST', 'mi2u-eInvoice'),
					'placeholder' => __('SST', 'mi2u-eInvoice'),
					'required'    => false,
				)); ?>
			</div>
			<br>
			<?php

			if (
				isset($_POST['ResubmitEInvoice']) && isset($_POST['mi2uinv_resubmit_nonce']) &&
				wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mi2uinv_resubmit_nonce'])), 'mi2uinv_resubmit_action')
			) {
				if (array_key_exists('ResubmitEInvoice', $_POST))
					$this->ResubmitEInvoice($order);
			}
			// if (array_key_exists('ResubmitEInvoice', $_POST)) {
			// 	$this->ResubmitEInvoice($order);
			// }
			?>
			<input type="submit" name="ResubmitEInvoice"
				class="button" value="Submit" />
		</form>

		<script type="text/javascript">
			function myFunction() {

			}

			jQuery(document).ready(function($) {
				$("input:radio").change(function() {
					// Do something interesting here
					if (this.value == 'NRIC' || this.value == 'ARMY') {
						if (document.getElementById("mi2uinv_tin").value == '' ||
							document.getElementById("mi2uinv_tin").value == 'EI00000000020')
							document.getElementById("mi2uinv_tin").value = 'EI00000000010';
						//alert(this.value);
					} else if (this.value == 'BRN') {
						if (document.getElementById("mi2uinv_tin").value == 'EI00000000010')
							document.getElementById("mi2uinv_tin").value = '';
						//alert(this.value);
					} else if (this.value == 'PASSPORT' || this.value == 'MyPR' || this.value == 'MyKAS') {
						if (document.getElementById("mi2uinv_tin").value == '' ||
							document.getElementById("mi2uinv_tin").value == 'EI00000000010')
							document.getElementById("mi2uinv_tin").value = 'EI00000000020';
						//alert(this.value);
					}
				});
			});
		</script>
	<?php
	}

	function ResubmitEInvoice($order)
	{
		if (
			isset($_POST['ResubmitEInvoice']) && isset($_POST['mi2uinv_resubmit_nonce']) &&
			wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mi2uinv_resubmit_nonce'])), 'mi2uinv_resubmit_action')
		) {
		} else {
			return;
		}
		if (isset($_POST['mi2uinv_comp']) && ! empty($_POST['mi2uinv_comp'])) {
			//$order->add_meta_data('up_ics', 'f', true);		
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_comp');
			$order->add_meta_data('mi2uinv_comp', sanitize_text_field(wp_unslash($_POST['mi2uinv_comp'])), true);
			$order->save_meta_data();
		}

		if (isset($_POST['mi2uinv_tin']) && ! empty($_POST['mi2uinv_tin'])) {
			//$order->add_meta_data('up_ics', 'f', true);		
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_tin');
			$order->add_meta_data('mi2uinv_tin', sanitize_text_field(wp_unslash($_POST['mi2uinv_tin'])), true);
			$order->save_meta_data();
		}

		if (isset($_POST['mi2uinv_idtype']) && ! empty($_POST['mi2uinv_idtype'])) {
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_idtype');
			$order->add_meta_data('mi2uinv_idtype', sanitize_text_field(wp_unslash($_POST['mi2uinv_idtype'])), true);
			$order->save_meta_data();
		}

		if (isset($_POST['mi2uinv_idvalue']) && ! empty($_POST['mi2uinv_idvalue'])) {
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_idvalue');
			$order->add_meta_data('mi2uinv_idvalue', sanitize_text_field(wp_unslash($_POST['mi2uinv_idvalue'])), true);
			$order->save_meta_data();
		}

		if (isset($_POST['mi2uinv_sst']) && ! empty($_POST['mi2uinv_sst'])) {
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_sst');
			$order->add_meta_data('mi2uinv_sst', sanitize_text_field(wp_unslash($_POST['mi2uinv_sst'])), true);
			$order->save_meta_data();
		}

		if ((isset($_POST['mi2uinv_tin']) && ! empty($_POST['mi2uinv_tin'])) &&
			isset($_POST['mi2uinv_idvalue']) && ! empty($_POST['mi2uinv_idvalue'])
		) {
			$order->update_meta_data('up_ics', 'f');
			$order->save_meta_data();
		}
	}

	function mi2uinv_save_custom_checkout_field_value_order_meta($order_id)
	{
		if (
			!isset($_POST['mi2uinv_checkout_nonce']) ||
			!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mi2uinv_checkout_nonce'])), 'mi2uinv_checkout_action')
		) {
			return;
		}

		write_log('mi2uinv_save_custom_checkout_field_value_order_meta ' . $order_id);
		$order = wc_get_order($order_id);
		if (isset($_POST['mi2uinv_tin']) && ! empty($_POST['mi2uinv_tin'])) {
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_tin');
			$order->add_meta_data('mi2uinv_tin', sanitize_text_field(wp_unslash($_POST['mi2uinv_tin'])), true);
			$order->save_meta_data();
		}

		if (isset($_POST['mi2uinv_idtype']) && ! empty($_POST['mi2uinv_idtype'])) {
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_idtype');
			$order->add_meta_data('mi2uinv_idtype', sanitize_text_field(wp_unslash($_POST['mi2uinv_idtype'])), true);
			$order->save_meta_data();
		}

		if (isset($_POST['mi2uinv_idvalue']) && ! empty($_POST['mi2uinv_idvalue'])) {
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_idvalue');
			$order->add_meta_data('mi2uinv_idvalue', sanitize_text_field(wp_unslash($_POST['mi2uinv_idvalue'])), true);
			$order->save_meta_data();
		}

		if (isset($_POST['mi2uinv_sst']) && ! empty($_POST['mi2uinv_sst'])) {
			write_log('mi2uinv_save_custom_checkout_field_value_order_meta mi2uinv_sst');
			$order->add_meta_data('mi2uinv_sst', sanitize_text_field(wp_unslash($_POST['mi2uinv_sst'])), true);
			$order->save_meta_data();
		}
	}

	function mi2uinv_custom_checkout_field($checkout)
	{
		wp_nonce_field('mi2uinv_checkout_action', 'mi2uinv_checkout_nonce');
	?>
		<div id="custom_checkout_field_display">
			<h3><?php esc_attr_e('E-Invoice', 'mi2u-eInvoice'); ?></h3>
			<h5 style="color:red"><?php esc_attr_e('Please fill-up TIN, IDType and IDValue if you need an e-invoice', 'mi2u-eInvoice'); ?></h5>

			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_tin', array(
					'type'        => 'text',
					'class'       => array('form-row-wide'),
					'label'       => __('TIN', 'mi2u-eInvoice'),
					'placeholder' => __('TIN', 'mi2u-eInvoice'),
					'required'    => false,
					'default' => 'EI00000000010'
				), $checkout->get_value('mi2uinv_tin')); ?>
			</div>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_idtype', array(
					'type'        => 'radio',
					'class'       => array('form-row-wide'),
					'label'       => __('IDType', 'mi2u-eInvoice'),
					'placeholder' => __('IDType', 'mi2u-eInvoice'),
					'required'    => false,
					'options'     => array(
						'NRIC' => __('NRIC', 'mi2u-eInvoice'),
						'BRN' => __('BRN', 'mi2u-eInvoice'),
						'PASSPORT' => __('PASSPORT', 'mi2u-eInvoice'),
						'ARMY' => __('ARMY', 'mi2u-eInvoice'),
						'MyPR' => __('MyPR', 'mi2u-eInvoice'),
						'MyKAS' => __('MyKAS', 'mi2u-eInvoice'),
					),
					'default' => 'NRIC'
				), $checkout->get_value('mi2uinv_idtype')); ?>
			</div>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_idvalue', array(
					'type'        => 'text',
					'class'       => array('form-row-wide'),
					'label'       => __('IDValue', 'mi2u-eInvoice'),
					'placeholder' => __('IDValue', 'mi2u-eInvoice'),
					'required'    => false,
				), $checkout->get_value('mi2uinv_idvalue')); ?>
			</div>
			<div class="custom-text-field-wrapper">
				<?php woocommerce_form_field('mi2uinv_sst', array(
					'type'        => 'text',
					'class'       => array('form-row-wide'),
					'label'       => __('SST', 'mi2u-eInvoice'),
					'placeholder' => __('SST', 'mi2u-eInvoice'),
					'required'    => false,
				), $checkout->get_value('mi2uinv_sst')); ?>
			</div>
		</div>

		<script type="text/javascript">
			jQuery(document).ready(function($) {
				$("input:radio").change(function() {
					// Do something interesting here
					if (this.value == 'NRIC' || this.value == 'ARMY') {
						if (document.getElementById("mi2uinv_tin").value == '' ||
							document.getElementById("mi2uinv_tin").value == 'EI00000000020')
							document.getElementById("mi2uinv_tin").value = 'EI00000000010';
						//alert(this.value);
					} else if (this.value == 'BRN') {
						if (document.getElementById("mi2uinv_tin").value == 'EI00000000010')
							document.getElementById("mi2uinv_tin").value = '';
						//alert(this.value);
					} else if (this.value == 'PASSPORT' || this.value == 'MyPR' || this.value == 'MyKAS') {
						if (document.getElementById("mi2uinv_tin").value == '' ||
							document.getElementById("mi2uinv_tin").value == 'EI00000000010')
							document.getElementById("mi2uinv_tin").value = 'EI00000000020';
						//alert(this.value);
					}
				});
			});
		</script>
	<?php
	}

	function mi2u_menu()
	{
		//$svg_icon = base64_encode(file_get_contents(MI2UINV_ASSETS . 'images/icon.svg'));
		add_menu_page(
			'E-Invoice MI2U',
			'E-Invoice MI2U',
			'manage_options',
			'mg-pi',
			__NAMESPACE__ . '\\plugin_page',
		);

		add_submenu_page('mg-pi', 'Main', 'MI2U Main', 'manage_options', 'custom-slug-main', array($this, 'main_page'));
		add_submenu_page('mg-pi', 'Name of page', 'Topup Credit', 'manage_options', 'custom-slug', array($this, 'topup_credit'));
	}

	function main_page()
	{
		echo '<script type="text/javascript">
           window.open("' . esc_url($this->GetWebSite()) . '/woocommerce")
      	</script>';
	}

	function topup_credit()
	{
		$token = get_option('mi2uinv_login_email');
		$passwordUser = base64_decode($token);
		$reqDataStr = json_decode($passwordUser);
		$email = $reqDataStr->username;

		$link = $this->GetWebSite() . "/topup/" . $email;
		echo '<script type="text/javascript">
           window.open("' . esc_url($link) . '")
        </script>';
	}

	function mi2u_profile_fields($user)
	{
		// let's get custom field values
		$IDType = ($IDType = get_user_meta($user->ID, 'IDType', true)) ? $IDType : 'BRN';
		$IDValue = get_user_meta($user->ID, 'IDValue', true);

		$TIN = get_user_meta($user->ID, 'TIN', true);

	?>
		<h3>Additional Information</h3>
		<table class="form-table">
			<tr>
				<th><label for="TIN">TIN</label></th>
				<td>
					<input type="text" name="TIN" id="TIN" value="<?php echo esc_attr($TIN) ?>" class="regular-text" />
				</td>
			</tr>
			<tr>
				<th><label for="IDType">ID Type</label></th>
				<td>
					<select id="IDType" name="IDType">
						<?php echo '<option value="' . 'NRIC' . '"' . selected('NRIC', $IDType, false) . '>' . esc_html('NRIC') . '</option>';
						?>
						<?php echo '<option value="' . 'BRN' . '"' . selected('BRN', $IDType, false) . '>' . esc_html('Business Registration Number') . '</option>';
						?>
					</select>
				</td>
			</tr>
			<tr>
				<th><label for="IDValue">ID Value</label></th>
				<td>
					<input type="text" name="IDValue" id="IDValue" value="<?php echo esc_attr($IDValue) ?>" class="regular-text" />
				</td>
			</tr>
		</table>
<?php
	}

	function mi2uinv_custom_action_function()
	{
		// Send the email
		if (false === as_next_scheduled_action('mi2u_check_order')) {
			//as_schedule_recurring_action(time(), 3600, 'mi2u_check_order');
			as_schedule_recurring_action(time(), 60, 'mi2u_check_order');
		}
	}

	function GetWebSite()
	{
		//return 'http://seedraw.ddns.net:1234/myinvoice2u/public';
		$web = get_option('mi2uinv_website_option');
		if ($web == 'add_uat') {
			return MI2UINV_WEBPORTAL_UAT;
		} else
			return MI2UINV_WEBPORTAL;
	}

	function GetTransaction($token, $orderId)
	{
		$url = $this->GetWebSite() . '/wooSubmit'; // Replace with your API endpoint
		$data = [
			'login' => $token,
			'interfaceCode' => '202',
			'content' => $orderId,
			'returnCode' => '',
			'returnMsg' => ''
		];
		$jsonData = wp_json_encode($data);
		$resp = $this->callPostApi($url, $jsonData);
		return $resp;
	}

	function mi2uinv_check_order_do_action()
	{
		//$timezone=wp_timezone_string();
		//$testCode= WC()->countries->get_base_postcode();
		//$test = base64_decode('eyJ1c2VybmFtZSI6Imt1YW5tZW5nQG15aW52b2ljZTJ1LmNvbSIsInB3ZCI6InNlUXdEIzhGIn0=');
		$tax_classes = WC_Tax::get_tax_classes();
		$token = get_option('mi2uinv_login_email');
		$passwordUser = base64_decode($token);
		// write_log('login_email ' . $passwordUser);
		$reqDataStr = json_decode($passwordUser);
		// write_log('mi2uinv_login_email ' . $token);

		$web = get_option('mi2uinv_website_option');

		//$resp = $this->GetTransaction($token, 154);

		$query = new WC_Order_Query(array(
			'status' => 'wc-completed',
			'orderby' => 'date',
			'order' => 'DESC',
			'return' => 'ids',
			'limit' => '10000',
		));
		$orders = $query->get_orders();

		foreach ($orders as $id) {
			$order = wc_get_order($id);
			if (!$order->meta_exists('up_ics')) {
				$order->add_meta_data('up_ics', 'f', true);
				$order->save_meta_data();
			}
			$up_ics = $order->get_meta('up_ics');
			if ($up_ics == 'f') {
				$token = get_option('mi2uinv_login_email');
				$passwordUser = base64_decode($token);
				write_log('login_email ' . $passwordUser);
				$reqDataStr = json_decode($passwordUser);

				$email = $reqDataStr->username;

				//$alldata = $order->get_data();

				$invoice = new mi2u_invoice();
				$invoice->localDateTime = $order->get_data()['date_created'];
				$invoice->invoiceCodeNumber = $id;

				$invoice->invoiceTypeCode = "01";
				$invoice->invoiceCurrencyCode = "MYR";
				$invoice->totalExcludingTax = number_format(abs($order->get_total() - $order->get_total_tax()), 2, '.', '');
				$invoice->totalIncludingTax = number_format(abs($order->get_total()), 2, '.', '');
				$invoice->totalPayableAmount = number_format(abs($order->get_total()), 2, '.', '');
				$invoice->issueMode = '2';

				$invoice->supplier = [
					'supplierAddress' => [
						'addressLine0' => 'Set At ICS',
						'addressLine1' => '',
						'addressLine2' => '',
						'cityName' =>  'Set At ICS',
						'country' => 'MY',
						'postalZone' => '09020',
						'state' => '04',
					],
					'supplierBusinessActivityDescription' => "NA",
					'supplierContactNumber' => "0444444444",
					'supplierEmail' => $email,
					'supplierIndustryCode' => "00000",
					'supplierName' => "Ali",
					'supplierRegistration' => [
						'idType' => 'BRN',
						'idValue' => "NA",
						'tin' => "NA",
					],
				];

				$IsRefund = false;
				$orderOri = $order;
				if ($order->get_data()['parent_id'] != '0') {
					$invoice->localDateTime = $order->get_data()['date_created'];
					$orderOri = wc_get_order($order->get_data()['parent_id']);

					$IsRefund = true;

					$resp = $this->GetTransaction($token, $order->get_data()['parent_id']);
					if ($resp['status'] == 'success') {
						$data = json_decode($resp['content']);
						if (count($data->records)) {
							$record = $data->records[0];

							if ($record->issuedFlag == "101") {
								if ($record->invoiceStatus == "101") {
									$invoice->issueMode = '1';
								} else {
									$invoice->issueMode = '2';
								}
							} else {
								if (isset($record->invoiceStatus) && $record->invoiceStatus == "101") {
									$invoice->issueMode = '1';
								} else {
									$invoice->issueMode = '2';
								}
							}
							$invoice->invoiceTypeCode = "04";
							$invoice->originalInvoiceCodeNumber = $order->get_data()['parent_id'];
						} else {
							continue;
						}
					} else {
						continue;
					}
				}

				$comp = $orderOri->get_meta('mi2uinv_comp');
				$IDType = $orderOri->get_meta('mi2uinv_idtype');
				$IDValue = $orderOri->get_meta('mi2uinv_idvalue');
				$TIN = $orderOri->get_meta('mi2uinv_tin');
				$SST = $orderOri->get_meta('mi2uinv_sst');

				if ($comp == "") {
					$comp = $orderOri->get_billing_company();
					if ($comp == "") {
						$comp = $orderOri->get_billing_first_name() . ' ' . $orderOri->get_billing_last_name();
					}
				}
				//return;
				if ($TIN != '' && $IDValue != '') {
					if ($IDType == "0")
						$IDType = "NRIC";
					else if ($IDType == "1")
						$IDType = "BRN";
					else if ($IDType == "2")
						$IDType = "PASSPORT";
					else if ($IDType == "3")
						$IDType = "ARMY";
					else if ($IDType == "4")
						$IDType = "NRIC";
					else if ($IDType == "5")
						$IDType = "NRIC";

					$notes = [];
					$idType = 'NRIC';
					$idValue = 'NA';
					$tin = 'NA';
					if (!$IsRefund)
						$invoice->issueMode = '1';

					if ($IDType != "") {
						$idType = $IDType;
					}
					if ($IDValue != "") {
						$idValue = $IDValue;
					}
					if ($TIN != "") {
						$tin = $TIN;
					}
					if ($tin == "NA") {
						$idType = 'NRIC';
					}
					$buyerContactNumber = $orderOri->get_billing_phone();
					if ($buyerContactNumber == "") {
						$buyerContactNumber = '0123456789';
					}					

					$invoice->buyer = [
						'buyerAddress' => [
							'country' => $orderOri->get_billing_country(),
							'state' => $this->GetStateCode($orderOri->get_billing_state()),
							'cityName' => $orderOri->get_billing_city(),
							'addressLine0' => $orderOri->get_billing_address_1(),
							'addressLine1' => $orderOri->get_billing_address_2(),
							'addressLine2' => '',
						],
						'buyerEmail' => $orderOri->get_billing_email(),
						'buyerContactNumber' => $buyerContactNumber,
						'buyerName' =>  $comp,
						'buyerRegistration' => [
							'idType' => $idType,
							'idValue' => $idValue,
							'tin' => $tin,
							'sst' => $SST,
						],
					];
				} else {
					if (!$IsRefund)
						$invoice->issueMode = '2';
				}

				$lineItemArray = [];
				$lineTaxArrayAll = [];
				foreach ($order->get_items() as $key => $item) {

					$terms = get_the_terms($item->get_product_id(), 'product_cat');
					$total_line               = $item['line_total'];
					$tax                      = $item['line_tax'];
					$quantity                 = $item->get_quantity();
					$line_price_without_tax   = $total_line - $tax;
					$single_price_without_tax = $line_price_without_tax / max(1, $quantity);
					$tva_rate                 = $tax / max(1, $line_price_without_tax);

					$lineTaxArray = [];
					$totalTaxAmount = $item['total_tax'];

					if ($item['total_tax'] > 0) {
						$taxRate = $item['total_tax'] / $item['total'] * 100;
						$tax = [
							'taxType' => $this->GetTaxCode($item['tax_class']),
							'percent' => number_format(abs($taxRate), 2, '.', ''),
							'perUnitAmount' => 0,
							'measurement' => 'XUN',
							'quantity' => 1,
							'netAmount' => number_format(abs($item['total']), 2, '.', ''),
							'tax' => number_format(abs($item['total_tax']), 2, '.', ''),
						];
						array_push($lineTaxArray, $tax);
						array_push($lineTaxArrayAll, $tax);
					}

					$discountAmount = abs($item['subtotal'] - $item['total']);

					if ($item['total_tax']  == 0) {
						$tax = [
							'taxType' => '06',
							'percent' => 0,
							'perUnitAmount' => 0,
							'measurement' => 'XUN',
							'quantity' => 1,
							'netAmount' => number_format(abs($item['sub_total']), 2, '.', ''),
							'tax' => 0,
						];
						array_push($lineTaxArray, $tax);
					}
					$classCode = $this->GetClassificationCode($terms[0]->slug);

					$quantity = $item['quantity'];
					if ($quantity == 0)
						$quantity = 1;
					$Item = [
						'itemName' => $item['name'],
						'itemCode' => $item['product_id'],
						'measurement' => 'XUN',
						'quantity' => abs($item['quantity']),
						'totalExcludingTax' => number_format(abs($item['total']), 2, '.', ''),
						'unitPrice' => number_format(abs($item['subtotal'] / $quantity), 2, '.', ''),
						'discountAmount' => number_format($discountAmount, 2, '.', ''),
						'subtotal' => number_format(abs($item['subtotal']), 2, '.', ''),
						'classificationList' => [
							['code' => $classCode, 'group' => 'CLASS'],
						],
						'taxTotal' => [
							'totalTaxAmount' => number_format(abs($totalTaxAmount), 2, '.', ''),
							'taxSubTotalList' => $lineTaxArray,
						],
					];
					array_push($lineItemArray, $Item);
				}
				$invoice->invoiceLineItemList = $lineItemArray;

				if (count($lineTaxArrayAll) == 0) {
					$tax = [
						'taxType' => '06',
						'percent' => 0,
						'perUnitAmount' => 0,
						'measurement' => 'XUN',
						'quantity' => 0,
						'netAmount' => 0,
						'tax' => 0,
					];
					array_push($lineTaxArrayAll, $tax);
				}

				//Fill in Total Tax
				$invoice->taxTotal = [
					'totalTaxAmount' => number_format(abs($order->get_total_tax()), 2, '.', ''),
					'taxSubTotalList' => $lineTaxArrayAll
				];

				$ics = wp_json_encode($invoice);
				$test = $ics;

				$url = $this->GetWebSite() . '/wooSubmit'; // Replace with your API endpoint
				$data = [
					'login' => $token,
					'interfaceCode' => '201',
					'content' => $ics,
					'returnCode' => '',
					'returnMsg' => ''
				];
				$jsonData = wp_json_encode($data);
				$resp = $this->callPostApi($url, $jsonData);
				if ($resp['status'] == 'success') {
					$order->update_meta_data('up_ics', 't');
					$order->save_meta_data();
				} else {
				}
			}
		}
	}

	public function GetStateCode($code)
	{
		if ($code == 'Johor') {
			return '01';
		} else if ($code == 'KDH') {
			return '02';
		} else if ($code == 'KTN') {
			return '03';
		} else if ($code == 'MLK') {
			return '04';
		} else if ($code == 'NSN') {
			return '05';
		} else if ($code == 'PHG') {
			return '06';
		} else if ($code == 'PNG') {
			return '07';
		} else if ($code == 'PRK') {
			return '08';
		} else if ($code == 'PLS') {
			return '09';
		} else if ($code == 'SGR') {
			return '10';
		} else if ($code == 'TRG') {
			return '11';
		} else if ($code == 'SBH') {
			return '12';
		} else if ($code == 'SWK') {
			return '13';
		} else if ($code == 'KUL') {
			return '14';
		} else if ($code == 'LBN') {
			return '15';
		} else if ($code == 'PJY') {
			return '16';
		}
		return '10';
	}

	public function GetClassificationCode($class)
	{
		$classStr = explode('_', $class);
		if (count($classStr) > 0) {
			$code = $this->RemLeading($classStr[count($classStr) - 1]);
			for ($i = 0; $i <= 45; $i++) {
				$invID = str_pad($i, 3, '0', STR_PAD_LEFT);
				if ($code == $invID) {
					return $code;
				}
			}
		}
		return '022';
	}

	public function GetTaxCode($code)
	{
		$taxes = explode('-', $code);
		if (count($taxes) > 0) {
			$taxC = $this->RemLeading($taxes[count($taxes) - 1]);
			if ($taxC == '01' || $taxC == '02' || $taxC == '03' || $taxC == '04' || $taxC == '05')
				return $taxC;
		}

		return '01';
	}

	public function RemLeading($str)
	{
		return ltrim($str, ' ');
	}

	function callPostApi($url, $jsonData)
	{
		// // Initialize a cURL session
		// $ch = curl_init($url);

		// // Set cURL options
		// curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); // Return the response as a string
		// curl_setopt($ch, CURLOPT_POST, true); // Set request method to POST
		// curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonData); // Set the raw JSON data
		// curl_setopt($ch, CURLOPT_HTTPHEADER, [
		// 	'Content-Type: application/json', // Set content type to JSON
		// 	//'Authorization: Bearer ' . $token, // Set the Bearer token for authentication
		// ]);

		// // Execute the cURL request and capture the response
		// $response = curl_exec($ch);

		// // Check for errors
		// if (curl_errno($ch)) {
		// 	write_log($ch); //echo 'Error:' . curl_error($ch);			
		// } else {
		// 	// Optionally, decode the response if it's in JSON format
		// 	$responseData = json_decode($response, true);
		// }

		// // Close the cURL session
		// curl_close($ch);

		// return $responseData;

		$args = array(
			'body'        => $jsonData,
			'headers'     => array(
				'Content-Type' => 'application/json',
			),
			'timeout'     => 60,
			'redirection' => 5,
			'blocking'    => true,
			'httpversion' => '1.0',
			'sslverify'   => false,
		);

		$response = wp_remote_post($url, $args);

		if (is_wp_error($response)) {
			return array(
				'status' => 'error',
				'message' => $response->get_error_message()
			);
		}

		$body = wp_remote_retrieve_body($response);
		$responseData = json_decode($body, true);

		return $responseData;
	}

	public static function get_instance()
	{
		$class = get_called_class();
		if (! isset(self::$instances[$class])) {
			self::$instances[$class] = new $class();
		}

		return self::$instances[$class];
	}
}

define('MI2UINV_WEBPORTAL_UAT', 'https://agent-uat.myinvoice2u.com');
define('MI2UINV_WEBPORTAL', 'https://agent.myinvoice2u.com');
define('MI2UINV_VERSION', '0.2.3');
define('MI2UINV_PLUGIN_DIR', __DIR__);
define('MI2UINV_PLUGIN_FILE', __FILE__);
define('MI2UINV_INCLUDES', MI2UINV_PLUGIN_DIR . '/includes/');
define('MI2UINV_ASSETS', MI2UINV_PLUGIN_DIR . '/assets/');
define('MI2UINV_MIN_PHP_VER', '8.1');
define('MI2UINV_MIN_WC_VER', '7.0');
define(
	'MI2UINV_REQUIRED_EXTENSIONS',
	array(
		'gd',
		'mbstring',
		'iconv',
		'dom',
	)
);


require MI2UINV_PLUGIN_DIR . '/init-freemius.php';

/**
 * Gets the environment warning.
 *
 * @return bool|string  The environment warning or false if all good.
 */
function get_environment_warning()
{
	if (version_compare(phpversion(), MI2UINV_MIN_PHP_VER, '<')) {
		/* translators: 1: minimun php version number. 2: current php version number */
		$message = __('E-Invoicing For Woocommerce - The minimum PHP version required for this plugin is %1$s. You are running %2$s.', 'mi2u-eInvoice');
		return sprintf($message, MI2UINV_MIN_PHP_VER, phpversion());
	}

	foreach (MI2UINV_REQUIRED_EXTENSIONS as $name) {
		if (! extension_loaded($name)) {
			/* translators: 1: the required extension name */
			$message = __('E-Invoicing For Woocommerce requires the php extension "%s" to be loaded to work.', 'mi2u-eInvoice');
			return sprintf($message, strtoupper($name));
		}
	}

	if (! defined('WC_VERSION')) {
		return __('E-Invoicing For Woocommerce requires WooCommerce to be activated to work.', 'mi2u-eInvoice');
	}

	if (version_compare(WC_VERSION, MI2UINV_MIN_WC_VER, '<')) {
		/* translators: 1: minimun WooCommerce version number. 2: current WooCommerce version number */
		$message = __('E-Invoicing For Woocommerce - The minimum WooCommerce version required for this plugin is %1$s. You are running %2$s.', 'mi2u-eInvoice');
		return sprintf($message, MI2UINV_MIN_WC_VER, WC_VERSION);
	}

	return false;
}



/**
 * Loads the plugins files if no environment warning
 */
function load()
{
	define('MI2UINV_PLUGIN_BASENAME', plugin_basename(__FILE__));
	load_plugin_textdomain('einvoicing-for-woocommerce', false, dirname(MI2UINV_PLUGIN_BASENAME) . '/languages');
	$warning = get_environment_warning();

	define('MI2UINV_PLUGIN_ASSETS', untrailingslashit(plugins_url('/', __FILE__)) . '/assets/');

	if (is_admin()) {
		include MI2UINV_INCLUDES . '/admin.php';
		// woocommerce settings.
		add_filter(
			'woocommerce_get_settings_pages',
			function ($settings) {
				$settings[] = include MI2UINV_INCLUDES . '/class-settings.php';
				return $settings;
			}
		);
	}

	// Common includes done.

	/**
	 * Our plugin is ready, and we are in dashboard let the word know it
	 *
	 * @since 0.0.4
	 */
	do_action('mi2u_ready');
}

// Activation hook to enable onboarding notice.
register_activation_hook(
	__FILE__,
	function () {
		update_option('mi2u_version', MI2UINV_VERSION);
		set_transient('mi2u_just_activated', 'yes');
	}
);


// Load when all plugins are loaded so we can check for WooCommerce version.
// We load at 9, class-invoice-customizer.php need to setup correctly the customizer by hooking into the customize_loaded_components filter before the customizer inits (customizer inits at 10 : _wp_customize_publish_changeset and _wp_customize_include in wp-includes/themes.php).
add_action('plugins_loaded', __NAMESPACE__ . '\load', 9);
